<?php

if (!defined('ABSPATH')) die('No direct access allowed');

class UpdraftPlus_Temporary_Clone_Status {
	
	/**
	 * Constructor for the class.
	 */
	public function __construct() {
		add_action('init', array($this, 'init'));
	}

	/**
	 * This function is called via the WordPress init action, it will check if the page is not the admin backend and output the clone status
	 *
	 * @return void
	 */
	public function init() {
		if (is_admin() || (defined('WP_CLI') && WP_CLI) || 'GET' != $_SERVER['REQUEST_METHOD']) return;
		$this->page_start();
		echo '<div class="updraftclone_content_container">';
		echo '<img class="updraftclone_logo" alt="UpdraftClone Logo" src="data:image/png;base64,'.$this->get_uc_logo_base64().'">';
		echo '<h1>' . $this->get_status_title() . '</h1>';
		echo '<p>' . $this->get_status_description() . '</p>';
		echo $this->get_content();
		echo '</div>';
		$this->page_end();
		die();
	}

	/**
	 * This function will output the start of the updraftclone status page
	 *
	 * @return void
	 */
	public function page_start() {
		echo '<!DOCTYPE html>
		<html xmlns="http://www.w3.org/1999/xhtml" class="wp-toolbar" lang="en-US">
		<head>
		<meta http-equiv="Content-Type" content="text/html; charset=UTF-8">
		<meta http-equiv="refresh" content="60">
		<meta name="robots" content="noindex, nofollow">
		<title>UpdraftClone</title>
		<style>
			body {
				background-color: #db6939;
			}
			.updraftclone_content_container {
				margin:auto; 
				margin-top:40px; 
				width:80%; 
				text-align:center; 
				color: #ffffff; 
				font-family: Source Sans Pro, Helvetica, Arial, Lucida, sans-serif; 
				font-weight: 300; 
				font-size: 20px;
			}
			.updraftclone_logo {
				width: 40%;
			}
		</style>
		</head>
		<body>';
	}

	/**
	 * This function will output the end of the updraftclone status page
	 *
	 * @return void
	 */
	public function page_end() {
		echo '</body>
		</html>';
	}

	/**
	 * Get base64 encoded string of UpdraftPlus logo
	 *
	 * @return string - base64 encoded string of UpdraftPlus logo
	 */
	public function get_uc_logo_base64() {
		return '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';
	}


	/**
	 * This function will get and return the clone status title ready to be displayed on the page
	 *
	 * @return string - the clone status title
	 */
	public function get_status_title() {
		
		$status = $this->get_status();
		$code = "";

		switch ($status) {
			case 'Installed':
				$code = "WordPress installed";
				break;
			case 'Uploading':
				$code = "Uploading backup";
				break;
			case 'Restoring':
				$code = "Restoring backup";
				break;
			default:
				$code = "";
				break;
		}

		return $code;
	}

	/**
	 * This function will get and return the clone status description ready to be displayed on the page
	 *
	 * @return string - the clone status description
	 */
	public function get_status_description() {
		global $updraftplus;

		$status = $this->get_status();
		$description = "";

		switch ($status) {
			case 'Installed':
				$description = "WordPress installed; now awaiting the site data";
				break;
			case 'Uploading':
				$backup_details = $this->get_backup_details();
				$description = "The sending of the backup set has begun. So far {$backup_details['sets']} data archives with a total of {$backup_details['uploaded']} MB have been uploaded";
				break;
			case 'Restoring':
				UpdraftPlus_Backup_History::rebuild();
				$backup_details = $this->get_backup_details();
				$description = "The site data has all been received, and its import has begun: {$backup_details['sets']} data archives remain";
				break;
			default:
				$description = "";
				break;
		}

		return $description;
	}

	/**
	 * This function will return information about the backup such as the amount of sets and the size of the backup set
	 *
	 * @return array - an array with backup information
	 */
	public function get_backup_details() {
		global $updraftplus;
		
		$backup_history = UpdraftPlus_Backup_History::get_history();
		$backupable_entities = $updraftplus->get_backupable_file_entities();
		$sets = 0;
		$uploaded = 0;
		
		foreach ($backupable_entities as $key => $info) {
			foreach ($backup_history as $timestamp => $backup) {
				if (isset($backup[$key]) && isset($backup[$key.'-size'])) {
					$sets += count($backup[$key]);
					$uploaded += $backup[$key.'-size'];
				}
			}
		}
		
		$uploaded = round($uploaded / 1048576, 1);

		return array('uploaded' => $uploaded, 'sets' => $sets);
	}

	/**
	 * This function will get and return the clone content ready to be displayed on the page
	 *
	 * @return string - the clone content
	 */
	public function get_content() {
		$content = "<p>Your UpdraftClone is still setting up. You can check the progress here or <a href='https://updraftplus.com/my-account/clones/'>in your UpdraftPlus.com account</a>.</p>";
		$content .= "<p>To read FAQs/documentation about UpdraftClone, <a href='https://updraftplus.com/faq-category/updraftclone/'>go here</a>.</p>";

		return $content;
	}

	/**
	 * This function will work out what stage the clone is in and return the correct status code
	 *
	 * @return string - the clone status code
	 */
	public function get_status() {
		global $updraftplus;

		$backup_history = UpdraftPlus_Backup_History::get_history();

		if (empty($backup_history)) return 'Installed';

		$updraft_dir = trailingslashit($updraftplus->backups_dir_location());

		if (file_exists($updraft_dir.'ready_for_restore')) return 'Restoring';

		return 'Uploading';
	}
}

if (defined('UPDRAFTPLUS_THIS_IS_CLONE') && 1 == UPDRAFTPLUS_THIS_IS_CLONE) {
	$updraftplus_temporary_clone_status = new UpdraftPlus_Temporary_Clone_Status();
}
