<?php
/*
Copyright (c) 2010-2018 Box Hill LLC

All Rights Reserved

No part of this software may be reproduced, copied, modified or adapted, without the prior written consent of Box Hill LLC.

Commercial use and distribution of any part of this software is not allowed without express and prior written consent of Box Hill LLC.


 */

/**
 * Class EasyRecipeScheduler
 *
 * WP Cron scheduler helper
 */
class EasyRecipeScheduler {

    /**
     * Schedule a run even if there is a run already scheduled
     */
    const FORCE_RUN = 1;
    /**
     *
     */
    const OVERRIDE_TIMEOUT = 2;
    const FORCE_CRON = 4;

    private $cronHook;
    private $timestamp;
    private $args = array();

    /**
     * @var int The "run" timeout. Need to store this so we can workaround the WP bug in get_transient
     */
    private $timeout = 0;

    /** @var  EasyLoggerLog */
    private $logger;

    /**
     * @param $cronHook
     */
    function __construct($cronHook) {
        $this->log("create $cronHook");
        $this->cronHook = $cronHook;
    }

    /**
     * Add an action for the CRON job
     * @param $method
     */
    function addAction($method) {
        $this->log("$this->cronHook: add action");
        add_action($this->cronHook, $method);
    }

    /**
     * Schedule a run at the head of any existing crob jobs so that this task will be the first to be executed,
     * even if there are existing tasks on the crom queue that would otherwise get run first
     * @param int $options
     * @param array $args
     * @return bool Returns TRUE if the job was scheduled else FALSE
     */
    function runImmediate($options = 0, $args = array()) {
        $crons = _get_cron_array();
        if ($crons == false) {
            return false;
        }

        /**
         * If there's something on the queue that's ready to run, schedule our job ahead of it
         */
        $keys = array_keys($crons);
        $firstJob = isset($keys[0]) ? $keys[0] : microtime(true);
        $time = min(time(), $firstJob);

        return $this->runAt($time - 1, $options | self::OVERRIDE_TIMEOUT, $args);
    }

    /**
     * Schedule a run NOW unless it's already scheduled or running
     * This simply adds the task to the cron transients to be run.  There may be other tasks already scheduled that will get run before this task
     *
     * @param int $options
     * @param array $args
     * @return bool Returns TRUE if the job was scheduled else FALSE
     */
    function runNow($options = 0, $args = array()) {
        $this->log("$this->cronHook: run now");
        return $this->runAt(time(), $options, $args);
    }

    /**
     * Schedule a run at time "$time" unless a it's already scheduled or running
     *
     * @param integer $time Unix timestamp
     * @param int $options
     * @param array $args A positional array of argument to be passed to the job to be run. $args[0] will be arg1, $args[1] will be arg2 etc
     *
     * @return bool Returns TRUE if the job was scheduled else FALSE
     */
    function runAt($time, $options = 0, $args = array()) {
        /**
         * Run even if there there is a run of this job already scheduled or running?
         */
        $forceRun = $options & self::FORCE_RUN;
        /**
         * Run even if the default timeout between cron runs (60 secs) hasn't passed since the last WP cron run ?
         */
        $overrideTimeout = $options & self::OVERRIDE_TIMEOUT;

        $gmtTime = time();
        $nextScheduled = $this->isScheduled($args);
        if ($forceRun || !$nextScheduled) {
            if ($forceRun || !$this->isRunning($args)) {
                $this->log("$this->cronHook: schedule at $time (time() is $gmtTime)");
                $this->timestamp = $time;
                $this->args = $args;
                wp_schedule_single_event($time, $this->cronHook, $args);
                /**
                 * If we want to run this NOW, then kick off the cron process
                 */
                if ($time <= time()) {
                    $gmtTime = $overrideTimeout ? microtime(true) + 60 : 0;
                    $this->log("$this->cronHook: spawn_cron($gmtTime)");
                    spawn_cron($gmtTime);
                }
                return true;
            } else {
                $this->log("$this->cronHook: schedule at $time but is already running");
            }
        } else {
            /**
             * The job is already scheduled - if it's due to run, then do it
             */
            $this->log("$this->cronHook: schedule at $time but is already scheduled for $nextScheduled (time() is $gmtTime)");
            if ($nextScheduled <= time()) {
                $gmtTime = $overrideTimeout ? microtime(true) + 60 : 0;
                $this->log("$this->cronHook: spawn_cron($gmtTime)");
                spawn_cron($gmtTime);
            }
        }
        return false;
    }

    /**
     * Returns TRUE if a run is already scheduled
     * Can't always do this via WP get_transient() because of a bug that deletes an autoloaded transient if it was set after this process started (and read autoloaded transients)
     * https://core.trac.wordpress.org/changeset/33110
     *
     * It's also better to check the DB anyway because it's possible that the process that does this check started before the transient was set by another process, but does this check after it was set
     * If we rely on get_option() in that case, it will incorrectly report the transient not set (because it will have been autoloaded at process startup)
     * Using get_option() to get the transiemt timeout is not a problem if there's a timeout, because timeout transients aren't autoloaded
     *
     * @param array $args
     * @return bool
     */
    function isRunning($args = array()) {
        /** @var wpdb $wpdb */
        global $wpdb;

        $this->log("$this->cronHook: isRunning");
        /**
         * If there's a timeout, check to see if it's expired, and if so, delete the transient (and its timeout)
         */
        if ($this->timeout != 0) {
            $timeoutKey = '_transient_timeout_' . $this->transientKey($args);
            $timeout = get_option($timeoutKey);
            if ($timeout !== false && $timeout < time()) {
                $this->log("$this->cronHook: isRunning timed out");
                $key = '_transient_' . $this->transientKey($args);
                delete_option($key);
                delete_option($timeoutKey);
                return false;
            }
        }

        $key = '_transient_' . $this->transientKey($args);
        $q = "SELECT option_value FROM  {$wpdb->options} WHERE option_name = '$key'";
        $value = $wpdb->get_var($q);
        return $value === 'run';
    }

    /**
     * Returns the timestamp of the next scheduled run or FALSE if there is no scheduled run.
     *
     * @param array $args The args of the event to be checked
     * @return bool|int
     */
    function isScheduled($args = array()) {
        $this->log("$this->cronHook: isScheduled");
        if (empty($args)) {
            $args = $this->args;
        }
        return wp_next_scheduled($this->cronHook, $args);
    }


    /**
     * Sets the status of this CRON as "running" with an optional timeout
     *
     * @param int $timeout The job will be marked as "running" for this many seconds, unless terminate() is called first
     */
    function setRunning($timeout = 0) {
        $this->log("$this->cronHook: set running");
        $this->timeout = $timeout;
        set_transient($this->transientKey(), 'run', $timeout);
    }

    /**
     * Mark a run as terminated, and/or cancel a run by removing the 'run' status (terminate) and removing the event (cancel) in case it never actually ran
     * @param array $args
     */
    function terminate($args = array()) {
        $this->log("$this->cronHook: terminate");
        if (empty($args)) {
            $args = $this->args;
        }
        delete_transient($this->transientKey($args));

        wp_unschedule_event($this->timestamp, $this->cronHook, $args);
    }

    /**
     * Return a key we can use for the "running" transient that's unique for the hook and args
     *
     * @param array $args
     * @return string
     */
    private function transientKey($args = array()) {
        if (empty($args)) {
            $args = $this->args;
        }
        return empty($args) ? $this->cronHook : $this->cronHook . "-" . md5(serialize($args));
    }

    /**
     * Debug logger
     *
     * @param $msg
     */
    private function log($msg) {

        if (!isset($this->logger)) {
            $this->logger = EasyRecipeLogger::getLog('scheduler');
        }
        $this->logger->info($msg);
    }

    /**
     * It's possible that an instance saved and then re-instated will have a logger class that's no longer available
     * If that's the case, unset the logger and it will be re-instantiated with an available class when it's used
     */
    public function __wakeup() {
        if ($this->logger instanceof __PHP_Incomplete_Class) {
            unset($this->logger);
        }
    }

}

