<?php

/**
 * The class responsible for plugin upgrade procedures.
 *
 * @since        5.0.0
 * @package      Shortcodes_Ultimate
 * @subpackage   Shortcodes_Ultimate/includes
 */
final class Shortcodes_Ultimate_Upgrade {

	/**
	 * The current version of the plugin.
	 *
	 * @since    5.0.0
	 * @access   private
	 * @var      string    $current_version   The current version of the plugin.
	 */
	private $current_version;

	/**
	 * Name of the option which stores plugin version.
	 *
	 * @since    5.0.0
	 * @access   private
	 * @var      string    $saved_version_option   Name of the option which stores plugin version.
	 */
	private $saved_version_option;

	/**
	 * The full path of the upgrade file.
	 *
	 * @since    5.4.0
	 * @access   private
	 * @var      string    $upgrade_path   The full path of the upgrade file.
	 */
	private $upgrade_path;

	/**
	 * Define the functionality of the updater.
	 *
	 * @since   5.0.0
	 * @param string  $plugin_version The current version of the plugin.
	 */
	public function __construct( $plugin_version ) {

		$this->current_version      = $plugin_version;
		$this->saved_version_option = 'su_option_version';
		$this->upgrade_path         = '';

	}

	/**
	 * Run upgrades if version changed.
	 *
	 * @since  5.0.0
	 */
	public function maybe_upgrade() {

		if ( ! $this->is_version_changed() ) {
			return;
		}

		$this->maybe_upgrade_to( '5.0.0' );
		$this->maybe_upgrade_to( '5.0.7' );
		$this->maybe_upgrade_to( '5.1.1' );
		$this->maybe_upgrade_to( '5.2.0' );
		$this->maybe_upgrade_to( '5.4.0' );

		$this->update_saved_version();

	}

	/**
	 * Conditional check if plugin was updated.
	 *
	 * @since  5.0.0
	 * @access private
	 * @return boolean True if plugin was updated, False otherwise.
	 */
	private function is_version_changed() {
		return $this->is_saved_version_less_than( $this->current_version );
	}

	/**
	 * Conditional check if previous version of the plugin less than passed one.
	 *
	 * @since  5.0.0
	 * @access private
	 * @return boolean True if previous version of the plugin less than passed one, False otherwise.
	 */
	private function is_saved_version_less_than( $version ) {

		return version_compare(
			get_option( $this->saved_version_option, 0 ),
			$version,
			'<'
		);

	}

	/**
	 * Helper function to register a new upgrade routine.
	 *
	 * @since 5.4.0
	 * @param string $version New version number.
	 */
	private function maybe_upgrade_to( $version ) {

		if ( ! $this->is_saved_version_less_than( $version ) ) {
			return;
		}

		$this->upgrade_path = su_get_plugin_path() . 'includes/upgrade/' . $version . '.php';

		if ( ! file_exists( $this->upgrade_path ) ) {
			return;
		}

		include $this->upgrade_path;

	}

	/**
	 * Save current version number.
	 *
	 * @since  5.0.0
	 * @access private
	 */
	private function update_saved_version() {
		update_option( $this->saved_version_option, $this->current_version, false );
	}

}
